<?php

namespace DynamicContentForElementor;

class Features {
	/**
	 * all main features that can be activated/deactivated independently.
	 */
	public $all_features;
	/**
	 * @var array<string,array<string,string>>
	 */
	public $bundled_features;
	private $features_status = false;
	const FEATURES_STATUS_OPTION = 'dce_features_status_option';

	public function __construct() {
		$this->all_features = self::get_all_features_data();
		$status = $this->get_features_status();
		$this->bundled_features = self::get_bundled_features_data();
		$this->update_all_features_with_new_status( $status );
		$this->save_later_if_empty();
	}

	/**
	 * Avoid saving immediately to avoid races with db upgrades.
	 *
	 * @return void
	 */
	public function save_later_if_empty() {
		$option_json = get_option( self::FEATURES_STATUS_OPTION );
		if ( $option_json === false ) {
			if ( ! wp_next_scheduled( 'dce_auto_save_features' ) ) {
				wp_schedule_single_event( time() + 15 * 60, 'dce_auto_save_features' );
			}
			add_action( 'dce_save_features', function() {
				update_option( self::FEATURES_STATUS_OPTION, wp_json_encode( $this->get_features_status() ) );
			} );
		}
	}

	/**
	 * Get Widgets Groups
	 *
	 * @return array<string,string>
	 */
	public static function get_widgets_groups() {
		return [
			'LIST' => __( 'List', 'dynamic-content-for-elementor' ),
			'FAVORITES' => __( 'Favorites', 'dynamic-content-for-elementor' ),
			'MAPS' => __( 'Maps', 'dynamic-content-for-elementor' ),
			'PDF' => 'PDF',
			'CONTENT' => __( 'Content', 'dynamic-content-for-elementor' ),
			'CREATIVE' => __( 'Creative', 'dynamic-content-for-elementor' ),
			'DEV' => __( 'Developer', 'dynamic-content-for-elementor' ),
			'WOO' => 'WooCommerce',
			'ACF' => __( 'Advanced Custom Fields', 'dynamic-content-for-elementor' ),
			'METABOX' => 'Meta Box',
			'TOOLSET' => 'Toolset',
			'PODS' => 'Pods',
			'SVG' => 'SVG',
			'WEBGL' => 'WEBGL',
			'POST' => __( 'Post', 'dynamic-content-for-elementor' ),
		];
	}

	/**
	 * Get Extensions Groups
	 *
	 * @return array<string,string>
	 */
	public static function get_extensions_groups() {
		return [
			'COMMON' => __( 'for All Widgets', 'dynamic-content-for-elementor' ),
			'FREE' => __( 'for Elementor Free', 'dynamic-content-for-elementor' ),
			'PRO' => __( 'for Elementor Pro', 'dynamic-content-for-elementor' ),
			'FORM' => __( 'for Elementor Pro Form', 'dynamic-content-for-elementor' ),
		];
	}

	/**
	 * Get Dynamic Tags Groups
	 *
	 * @return array<string,string>
	 */
	public static function get_dynamic_tags_groups() {
		return [
			'POST' => __( 'Posts and Pages', 'dynamic-content-for-elementor' ),
			'WOO' => 'WooCommerce',
			'TERMS' => __( 'Terms', 'dynamic-content-for-elementor' ),
			'CONTENT' => __( 'Content', 'dynamic-content-for-elementor' ),
			'DEV' => __( 'Developer', 'dynamic-content-for-elementor' ),
			'ACF' => __( 'Advanced Custom Fields', 'dynamic-content-for-elementor' ),
			'METABOX' => 'Meta Box',
			'DYNAMIC_GOOGLE_MAPS_DIRECTIONS' => 'Dynamic Google Maps Directions',
		];
	}

	public function get_feature_info( string $feature_id, string $info ) {
		return $this->all_features[ $feature_id ][ $info ];
	}

	/**
	 * Get Feature Title
	 *
	 * @param string $feature_id
	 * @return string
	 */
	public function get_feature_title( string $feature_id ) {
		return $this->all_features[ $feature_id ][ 'title' ];
	}

	public function get_feature_info_by_array( array $features, string $key ) {
		$return = [];
		foreach ( $features as $feature ) {
			$return[] = $feature[ $key ];
		}
		return $return;
	}

	public function filter( array $args, string $operator = 'AND' ) {
		return wp_list_filter( $this->all_features, $args, $operator );
	}

	/**
	 * Filter Bundled
	 *
	 * @param array<mixed> $args
	 * @param string $operator
	 * @return array<mixed>
	 */
	public function filter_bundled( array $args, string $operator = 'AND' ) {
		return wp_list_filter( $this->bundled_features, $args, $operator );
	}

	public function filter_by_tag( string $value ) {
		return array_filter( $this->all_features, function( $f ) use ( $value ) {
			return in_array( $value, $f['tag'] ?? [], true );
		} );
	}

	/**
	 * Filter By Collection
	 *
	 * @param string $value
	 * @return array<string,array<string,string>>
	 */
	public function filter_by_collection( string $value ) {
		return array_filter( $this->all_features, function( $f ) use ( $value ) {
			return isset( $f['collection'] ) && $value === $f['collection'];
		} );
	}

	public function get_active_features() {
		return array_filter( $this->get_features_status(), function( $e ) {
			return 'active' === $e;
		} );
	}

	/**
	 * Is Feature Active
	 *
	 * @param string $feature
	 * @return boolean
	 */
	public function is_feature_active( string $feature ) {
		return $this->get_feature_info( $feature, 'status' ) === 'active';
	}

	public function update_all_features_with_new_status( $status ) {
		array_walk( $this->all_features, function( &$v, $k ) use ( $status ) {
			$v['status'] = $status[ $k ];
		});
		// bundled features should have the same status as the main features
		// they are bundled with:
		foreach( $this->bundled_features as $key => $data) {
			$status = $this->all_features[ $data['activated_by'] ]['status'];
			$this->bundled_features[$key]['status'] = $status;
		}
	}

	/**
	 * Return an associative array with the feature name as key and the value
	 * either 'active' or 'inactive' based on the status of the feature.
	 *
	 * @return array<string,string>
	 */
	public function get_features_status() {
		if ( $this->features_status === false ) {
			$option_json = get_option( self::FEATURES_STATUS_OPTION, false );
			if ( $option_json === false ) {
				// features never saved, use the default status.
				$default = array_map( function( $info ) {
					if ( ( $info['default_status'] ?? 'active' ) === 'inactive' ) {
						return 'inactive';
					}
					$is_legacy = isset( $info['legacy'] ) && $info['legacy'];
					return $is_legacy ? 'inactive' : 'active';
				}, $this->all_features );
				$this->features_status = $default;
			} else {
				// every feature that is not saved in the db should be inactive.
				$option = json_decode( $option_json, true );
				$default = array_map( function( $info ) {
					return 'inactive';
				}, $this->all_features );
				$this->features_status = $option + $default;
			}
		}
		return $this->features_status;
	}


	/**
	 * Received an assoc array with key feature name and value feature status.
	 * This can be a subset of all features. Update the feature option in the db with the provided values
	 * leaving unchanged the features not provided.
	 *
	 * @return void
	 */
	public function db_update_features_status( $subset_status ) {
		$status_before = array_map( function( $f ) {
			return $f['status'];
		}, $this->all_features );
		$new_status = $subset_status + $status_before;
		update_option( self::FEATURES_STATUS_OPTION, wp_json_encode( $new_status ) );
		$this->update_all_features_with_new_status( $new_status );
	}

	/**
	 * Get Bundled Features Data
	 *
	 * @return array<string,mixed>
	 */
	public static function get_bundled_features_data() {
		return [
			'ext_tag_dynamic_google_maps_directions_info' => [
				'activated_by' => 'wdg_google_maps_directions',
				'name' => 'ext_tag_dynamic_google_maps_directions_info',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\DynamicGoogleMapsDirectionsInfo',
				'category' => 'DYNAMIC_GOOGLE_MAPS_DIRECTIONS',
				'extension_type' => 'dynamic-tag',
				'title' => __( 'Dynamic Tag - Map Info', 'dynamic-content-for-elementor' ),
				'description' => __( 'Shows information from a map created with Dynamic Google Maps Directions', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-google-maps-directions-info',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/dynamic-tag-map-info',
			],
			'ext_tag_dynamic_google_maps_directions_instructions' => [
				'activated_by' => 'wdg_google_maps_directions',
				'name' => 'ext_tag_dynamic_google_maps_directions_instructions',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\DynamicGoogleMapsDirectionsInstructions',
				'category' => 'DYNAMIC_GOOGLE_MAPS_DIRECTIONS',
				'extension_type' => 'dynamic-tag',
				'title' => __( 'Dynamic Tag - Map Instructions', 'dynamic-content-for-elementor' ),
				'description' => __( 'Shows the road information of a map created with Dynamic Google Maps Directions', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-google-maps-directions-instructions',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/dynamic-tag-map-instructions',
			],
		];
	}

	/**
	 * Get All Features Data
	 *
	 * @return array<string,mixed>
	 */
	public static function get_all_features_data() {
		return [
			///  Extensions ///
			'ext_animations' => [
				'name' => 'dce_extension_animations',
				'type' => 'extension',
				'class' => 'Extensions\DynamicAnimations',
				'category' => 'COMMON',
				'title' => __( 'Dynamic Animations', 'dynamic-content-for-elementor' ),
				'description' => __( 'Predefined CSS-Animations with keyframe', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-animations',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/loop-animations/',
			],
			'ext_copypaste' => [
				'name' => 'dce_extension_copypaste',
				'type' => 'extension',
				'class' => 'Extensions\CopyPaste',
				'legacy' => true,
				'replaced_by_custom_message' => __( 'This feature has been set as deprecated since Elementor released its own "Copy and Paste Between Websites" functionality in v3.11.0.', 'dynamic-content-for-elementor' ),
				'category' => 'EDITOR',
				'title' => __( 'Copy&Paste Cross Sites', 'dynamic-content-for-elementor' ),
				'description' => __( 'Copy and Paste any element from one site to another', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-copy-paste',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/copy-paste-cross-site/',
			],
			'ext_masking' => [
				'name' => 'dce_extension_masking',
				'type' => 'extension',
				'class' => 'Extensions\AdvancedMasking',
				'category' => 'FREE',
				'title' => __( 'Advanced Masking Controls', 'dynamic-content-for-elementor' ),
				'description' => __( 'Advanced Masking features for Image, Image-box and Video Widgets', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-advanced-masking',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/advanced-masking/',
			],
			'ext_rellax' => [
				'name' => 'dce_extension_rellax',
				'type' => 'extension',
				'class' => 'Extensions\Rellax',
				'category' => 'COMMON',
				'title' => __( 'Rellax', 'dynamic-content-for-elementor' ),
				'description' => __( 'Rellax Parallax rules for widgets and rows', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-rellax',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/rellax-parallax/',
			],
			'ext_reveal' => [
				'name' => 'dce_extension_reveal',
				'type' => 'extension',
				'class' => 'Extensions\Reveal',
				'category' => 'COMMON',
				'title' => __( 'Reveal', 'dynamic-content-for-elementor' ),
				'description' => __( 'Reveal animation on-scroll for widgets', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-reveal',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/scroll-reveals/',
			],
			'ext_transforms' => [
				'name' => 'dce_extension_transforms',
				'type' => 'extension',
				'class' => 'Extensions\Transforms',
				'category' => 'COMMON',
				'title' => __( 'Transforms', 'dynamic-content-for-elementor' ),
				'description' => __( 'Apply CSS Transforms to Element', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-transforms',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/transforms/',
			],
			'ext_unwrap' => [
				'name' => 'dce_extension_unwrap',
				'default_status' => 'inactive',
				'type' => 'extension',
				'class' => 'Extensions\Unwrap',
				'category' => 'COMMON',
				'title' => __( 'Unwrap', 'dynamic-content-for-elementor' ),
				'description' => __( 'Remove extra Elementor wrapper divs', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-unwrap',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/unwrap/',
			],
			'ext_video' => [
				'name' => 'dce_extension_video',
				'type' => 'extension',
				'class' => 'Extensions\AdvancedVideoControls',
				'category' => 'FREE',
				'title' => __( 'Advanced Video Controls', 'dynamic-content-for-elementor' ),
				'description' => __( 'Advanced Video features for Elementor Video Widget', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-advanced-video-controls',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/advanced-video-controls/',
			],
			'ext_visibility' => [
				'name' => 'dce_extension_visibility',
				'type' => 'extension',
				'class' => 'Extensions\DynamicVisibility',
				'category' => 'COMMON',
				'title' => __( 'Dynamic Visibility', 'dynamic-content-for-elementor' ),
				'description' => __( 'Visibility rules for widgets, rows, columns, and sections', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-visibility',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/dynamic-visibility-for-elementor/',
			],
			'ext_form_address_autocomplete' => [
				'name' => 'dce_extension_form_address_autocomplete',
				'type' => 'extension',
				'class' => 'Extensions\AddressAutocomplete',
				'category' => 'FORM',
				'tag' => [ 'enchanted' ],
				'title' => __( 'Address Autocomplete for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'Use autocomplete to give your form fields the type-ahead-search behaviour of the Google Maps search field. The autocomplete service can match on full words and substrings, resolving place names and addresses', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-autocomplete-address',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/address-autocomplete-for-elementor-pro-form/',
			],
			'ext_form_amount' => [
				'name' => 'dce_extension_form_amount',
				'type' => 'extension',
				'class' => 'Extensions\Amount',
				'category' => 'FORM',
				'title' => __( 'Amount Field for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add an Amount Field to Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-amount',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/amount-elementor-pro-form/',
			],
			'ext_form_range' => [
				'name' => 'dce_extension_form_range',
				'type' => 'extension',
				'class' => 'Extensions\Range',
				'category' => 'FORM',
				'title' => __( 'Range Field for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add a Range Field to Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-range',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/range-field-elementor-pro-form/',
			],
			'ext_form_description' => [
				'name' => 'dce_extension_form_description',
				'type' => 'extension',
				'class' => 'Extensions\FieldDescription',
				'category' => 'FORM',
				'tag' => [ 'enchanted' ],
				'title' => __( 'Field Description for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'Describe your form field to help users better understand each field. You can add a tooltip or insert text', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-description',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/field-description-elementor-pro-form/',
			],
			'ext_form_email' => [
				'name' => 'dce_extension_form_email',
				'type' => 'extension',
				'class' => 'Extensions\DynamicEmail',
				'category' => 'FORM',
				'title' => __( 'Dynamic Email for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add Dynamic Email action to Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-email',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/dynamic-email-for-elementor-pro-form/',
			],
			'ext_form_favorites_action' => [
				'name' => 'dce_extension_form_favorites_action',
				'type' => 'extension',
				'class' => 'Extensions\FavoritesAction',
				'category' => 'FORM',
				'title' => __( 'Favorites Action for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add favorites via Elementor forms. Useful to use in combination with the "Add to Favorites" and "Favorites" widgets', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-favorites-action',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/favorites-elementor-pro-form/',
			],
			'ext_form_export' => [
				'name' => 'dce_extension_form_export',
				'type' => 'extension',
				'class' => 'Extensions\Export',
				'category' => 'FORM',
				'title' => __( 'Export for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add Export action to Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-export',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/export-for-elementor-pro-form/',
			],
			'ext_form_icons' => [
				'name' => 'dce_extension_form_icons',
				'type' => 'extension',
				'class' => 'Extensions\Icons',
				'category' => 'FORM',
				'tag' => [ 'enchanted' ],
				'title' => __( 'Icons for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add icons on the label or inside the input of form fields', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-icons',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/icons-for-elementor-pro-form/',
			],
			'ext_form_inline_align' => [
				'name' => 'dce_extension_form_inline_align',
				'type' => 'extension',
				'class' => 'Extensions\InlineAlign',
				'category' => 'FORM',
				'tag' => [ 'enchanted' ],
				'title' => __( 'Inline Align for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'Choose the inline align type for checkbox and radio fields', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-inline-align',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/inline-align-for-elementor-pro-form/',
			],
			'ext_form_length' => [
				'name' => 'dce_extension_form_length',
				'type' => 'extension',
				'class' => 'Extensions\FieldLength',
				'category' => 'FORM',
				'tag' => [ 'enchanted' ],
				'title' => __( 'Field Length for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'Choose a minimum and maximum characters length for your text and textarea fields', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-field-length',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/field-length-for-elementor-pro-form/',
			],
			'ext_form_message' => [
				'name' => 'dce_extension_form_message',
				'type' => 'extension',
				'class' => 'Extensions\MessageGenerator',
				'category' => 'FORM',
				'title' => __( 'Message Generator for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'Options to customize the Elementor Pro Form success message', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-message-generator',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/message-generator-for-elementor-pro-form/',
			],
			'ext_form_method' => [
				'name' => 'dce_extension_form_method',
				'type' => 'extension',
				'class' => 'Extensions\Method',
				'category' => 'FORM',
				'title' => __( 'Method for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add a different method attribute on your forms that specifies how to send form-data', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-method',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/method-for-elementor-pro-form/',
			],
			'ext_form_password_visibility' => [
				'name' => 'dce_extension_form_password_visibility',
				'type' => 'extension',
				'class' => 'Extensions\PasswordVisibility',
				'category' => 'FORM',
				'tag' => [ 'enchanted' ],
				'title' => __( 'Password Visibility for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'Allow your users to show or hide their password on Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-password-visibility',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/password-visibility-for-elementor-pro-form/',
			],
			'ext_form_unique_id' => [
				'name' => 'dce_form_unique_id',
				'type' => 'extension',
				'class' => 'Extensions\UniqueId',
				'category' => 'FORM',
				'title' => __( 'Unique ID Field for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'With this hidden field you can generate a unique ID to be used in PDF Generator or Dynamic Email', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-unique-id',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/unique-id-elementor-pro-form/',
			],
			'ext_form_counter' => [
				'name' => 'dce_form_counter',
				'type' => 'extension',
				'class' => 'Extensions\Counter',
				'category' => 'FORM',
				'title' => __( 'Counter Field for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'Perfect for keeping track of submissions or the sum of another form field', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-counter',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/counter-elementor-pro-form/',
			],
			'ext_form_pdf' => [
				'name' => 'dce_extension_form_pdf',
				'type' => 'extension',
				'class' => 'Extensions\PdfGenerator',
				'category' => 'FORM',
				'title' => __( 'PDF Generator for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add PDF creation action to Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-pdf-generator',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/pdf-generator-for-elementor-pro-form/',
			],
			'ext_form_php_action' => [
				'name' => 'php_action',
				'type' => 'extension',
				'class' => 'Extensions\PhpAction',
				'category' => 'FORM',
				'title' => __( 'PHP Action for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add PHP code to the actions of a form', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-form-php-action',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/php-action-elementor-pro-form/',
			],
			'ext_form_cookie_action' => [
				'name' => 'dce_cookie_action',
				'type' => 'extension',
				'class' => 'Extensions\CookieAction',
				'category' => 'FORM',
				'title' => __( 'Cookie Action for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'Set cookies after submitting the form', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-cookie-action',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/cookie-action-elementor-pro-form/',
			],
			'ext_form_redirect' => [
				'name' => 'dce_extension_form_redirect',
				'type' => 'extension',
				'class' => 'Extensions\DynamicRedirect',
				'category' => 'FORM',
				'title' => __( 'Dynamic Redirect for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'Redirect your users to different URLs based on their choice on submitted form fields', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-redirect',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/dynamic-redirect-for-elementor-pro-form/',
			],
			'ext_form_regex' => [
				'name' => 'dce_extension_form_regex',
				'type' => 'extension',
				'class' => 'Extensions\RegexField',
				'category' => 'FORM',
				'tag' => [ 'enchanted' ],
				'title' => __( 'Regex Field for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'Validate form fields using a regular expression', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-regex',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/regex-field-for-elementor-pro-form/',
			],
			'ext_form_reset' => [
				'name' => 'dce_extension_form_reset',
				'type' => 'extension',
				'class' => 'Extensions\ResetButton',
				'category' => 'FORM',
				'title' => __( 'Reset Button for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add a reset button which resets all form fields to their initial values', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-reset',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/reset-button-for-elementor-pro-form/',
			],
			'ext_form_save' => [
				'name' => 'dce_extension_form_save',
				'type' => 'extension',
				'class' => 'Extensions\Save',
				'category' => 'FORM',
				'title' => __( 'Save for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'Save the form data submitted by your client as posts, users or terms', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-save',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/save-elementor-pro-form/',
			],
			'ext_form_select2' => [
				'name' => 'dce_extension_form_select2',
				'type' => 'extension',
				'class' => 'Extensions\Select2',
				'category' => 'FORM',
				'tag' => [ 'enchanted' ],
				'title' => __( 'Select2 for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add Select2 to your select fields to gives a customizable select box with support for searching', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-select2',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/select2-for-elementor-pro-form/',
			],
			'ext_form_signature' => [
				'name' => 'dce_extension_form_signature',
				'type' => 'extension',
				'class' => 'Extensions\Signature',
				'category' => 'FORM',
				'title' => __( 'Signature for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add a signature field to Elementor Pro Form and it will be included in your PDF', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-signature',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/signature-field-for-elementor-pro-form',
			],
			'ext_form_step' => [
				'name' => 'dce_extension_form_step',
				'type' => 'extension',
				'class' => 'Extensions\EnhancedMultiStep',
				'category' => 'FORM',
				'title' => __( 'Enhanced Multi-Step for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add features to Elementor Pro Multi-Step: label as a legend, show all steps, scroll to top on step change and step summary', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-multistep',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/enhanced-multi-step-elementor-pro-form/',
				'legacy' => true,
			],
			'ext_form_submit_on_change' => [
				'name' => 'dce_extension_form_submit_on_change',
				'type' => 'extension',
				'class' => 'Extensions\SubmitOnChange',
				'category' => 'FORM',
				'tag' => [ 'enchanted' ],
				'title' => __( 'Submit On Change for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'Submit the form automatically when the user chooses a radio button or a select field', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-submit-on-change',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/submit-on-change-for-elementor-pro-form/',
			],
			'ext_form_submit' => [
				'name' => 'dce_extension_form_submit',
				'type' => 'extension',
				'class' => 'Extensions\SubmitButton',
				'category' => 'FORM',
				'title' => __( 'Submit Button for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add another submit button on your forms', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-submit',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/submit-button-for-elementor-pro-form/',
			],
			'ext_form_telegram' => [
				'name' => 'dce_extension_form_telegram',
				'type' => 'extension',
				'class' => 'Extensions\Telegram',
				'category' => 'FORM',
				'title' => __( 'Telegram for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'Send the content of your Elementor Pro Form to Telegram', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-telegram',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/telegram-for-elementor-pro-form/',
			],
			'ext_form_visibility' => [
				'name' => 'dce_extension_form_visibility',
				'type' => 'extension',
				'class' => 'Extensions\ConditionalFieldsOldVersion',
				'legacy' => true,
				'replaced_by' => 'ext_conditional_fields_v2',
				'category' => 'FORM',
				'title' => __( 'Conditional Fields (old version) for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'Conditionally display fields based on form field values', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-conditional-fields',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/conditional-fields-for-elementor-pro-form/',
			],
			'ext_conditional_fields_v2' => [
				'name' => 'dce_conditional_fields_v2',
				'type' => 'extension',
				'class' => 'Extensions\ConditionalFields',
				'category' => 'FORM',
				'title' => __( 'Conditionals for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'Contains the extensions: Conditional Fields, Conditional Validation, Conditional Mailchimp, Conditional Active Campaign, Conditional Webhook, Conditional MailerLite, and Max Submissions for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-conditional-fields',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/conditional-fields-v2-for-elementor-pro-form/',
			],
			'ext_form_wysiwyg' => [
				'name' => 'ext_form_wysiwyg',
				'type' => 'extension',
				'class' => 'Extensions\WYSIWYG',
				'category' => 'FORM',
				'title' => __( 'WYSIWYG Field for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add a WYSIWYG Field to your forms', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-wysiwyg',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/wysiwyg-field-elementor-pro-form/',
			],
			'ext_form_paypal' => [
				'name' => 'dce_extension_form_paypal',
				'type' => 'extension',
				'class' => 'Extensions\PaypalField',
				'category' => 'FORM',
				'title' => __( 'PayPal Field for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add a PayPal field for simple payments to Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-paypal',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/paypal-for-elementor-pro-form/',
			],
			'ext_form_stripe' => [
				'name' => 'dce_extension_form_stripe',
				'type' => 'extension',
				'class' => 'Extensions\StripeField',
				'category' => 'FORM',
				'title' => __( 'Stripe Field for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add a Stripe field for simple payments to Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-stripe',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/stripe-for-elementor-pro-form/',
			],
			'ext_custom_validation' => [
				'name' => 'dce_custom_validation',
				'type' => 'extension',
				'class' => 'Extensions\PhpValidation',
				'category' => 'FORM',
				'title' => __( 'PHP Validation for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add PHP code to validate a whole form', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-custom-validation',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/php-validation-for-elementor-pro-form/',
			],
			'ext_js_field' => [
				'name' => 'dce_js_field',
				'type' => 'extension',
				'class' => 'Extensions\JsField',
				'category' => 'FORM',
				'title' => __( 'JS Field for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'Take advantage of JS full power to set the value of a field on Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-js-amount',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/js-field-elementor-pro-form/',
			],
			'ext_formatted_number_field' => [
				'name' => 'dce_formatted_number_field',
				'type' => 'extension',
				'class' => 'Extensions\FormattedNumber',
				'category' => 'FORM',
				'title' => __( 'Formatted Number for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'A number field that is displayed formatted on blur', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-formatted-number',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/formatted-number-elementor-pro-form/',
			],
			'ext_live_html' => [
				'name' => 'dce_live_html',
				'type' => 'extension',
				'class' => 'Extensions\LiveHtml',
				'category' => 'FORM',
				'title' => __( 'Live HTML Field for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'Insert the values of your fields in an HTML Field and those changes will be updated live as they happen', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-live-html',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/live-html-field-elementor-pro-form/',
			],
			'ext_hidden_label' => [
				'name' => 'dce_hidden_label',
				'type' => 'extension',
				'class' => 'Extensions\HiddenLabel',
				'category' => 'FORM',
				'title' => __( 'Hidden Label for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'Get the label corresponding to the value of another Radio, Select or Checkbox field', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-hidden-label',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/hidden-label-elementor-pro-form/',
			],
			'ext_dynamic_select' => [
				'name' => 'dce_dynamic_select',
				'type' => 'extension',
				'class' => 'Extensions\DynamicSelect',
				'category' => 'FORM',
				'title' => __( 'Dynamic Select for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'Insert a select field where the list of options changes dynamically according to the value of another field', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-dynamic-select',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/dynamic-select-field-elementor-pro-form/',
			],
			'ext_tooltip' => [
				'name' => 'dce_tooltip',
				'type' => 'extension',
				'class' => 'Extensions\Tooltip',
				'category' => 'COMMON',
				'title' => __( 'Tooltip', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add a tooltip to any widget', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-tooltip',
				'plugin_depends' => [],
				'doc_url' => 'https://www.dynamic.ooo/widget/tooltip-elementor',
			],
			'ext_dynamic_countdown' => [
				'name' => 'dce_dynamic_countdown',
				'type' => 'extension',
				'class' => 'Extensions\DynamicCountdown',
				'category' => 'PRO',
				'legacy' => true,
				'replaced_by_custom_message' => __( 'This feature has been deprecated since Elementor Pro added new dynamic tag for due date in v3.10.0.', 'dynamic-content-for-elementor' ),
				'title' => __( 'Dynamic Due Date for Elementor Countdown', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add a Dynamic Due Date field on Elementor Countdown widget to permit dynamic values', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-dynamic-countdown',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/dynamic-due-date-for-elementor-countdown/',
			],
			'ext_confirm_dialog' => [
				'name' => 'dce_confirm_dialog',
				'type' => 'extension',
				'class' => 'Extensions\ConfirmDialog',
				'category' => 'FORM',
				'title' => __( 'Confirm Dialog for Elementor Pro Form', 'dynamic-content-for-elementor' ),
				'description' => __( 'An easy-to-use component that allows you to use Confirm/Cancel style dialogs on your form', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-confirm-dialog',
				'plugin_depends' => [ 'elementor-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/confirm-dialog-for-elementor-pro-form/',
			],

			///  Dynamic Tags ///

			'ext_tag_cryptocurrency' => [
				'name' => '',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\Cryptocurrency',
				'category' => 'CONTENT',
				'extension_type' => 'dynamic-tag',
				'title' => __( 'Dynamic Tag - Cryptocurrency', 'dynamic-content-for-elementor' ),
				'description' => __( 'The perfect way to use cryptocurrency on all widgets', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-tag-crypto',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/dynamic-tag-cryptocurrency',
			],
			'ext_tag_posts' => [
				'name' => 'ext_tag_posts',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\Posts',
				'category' => 'POST',
				'extension_type' => 'dynamic-tag',
				'title' => __( 'Dynamic Tag - Posts', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add support to show posts in Dynamic Tag for text settings', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-tag-posts',
				'plugin_depends' => '',
				'doc_url' => '',
			],
			'ext_tag_woo_products' => [
				'name' => 'ext_tag_woo_products',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\WooProducts',
				'category' => 'WOO',
				'extension_type' => 'dynamic-tag',
				'title' => __( 'Dynamic Tag - Products', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add support to show WooCommerce products in Dynamic Tag for text settings', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-tag-woo-products',
				'plugin_depends' => [ 'woocommerce' ],
				'doc_url' => '',
			],
			'ext_tag_sticky_posts' => [
				'name' => 'ext_tag_sticky_posts',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\StickyPosts',
				'category' => 'POST',
				'extension_type' => 'dynamic-tag',
				'title' => __( 'Dynamic Tag - Sticky Posts', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add support to show Sticky Posts in Dynamic Tag for text settings', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-tag-sticky-posts',
				'plugin_depends' => '',
				'doc_url' => '',
			],
			'ext_tag_my_posts' => [
				'name' => 'ext_tag_my_posts',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\MyPosts',
				'category' => 'POST',
				'extension_type' => 'dynamic-tag',
				'title' => __( 'Dynamic Tag - Posts by the Current User', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add support to show posts of the current user in Dynamic Tag for text settings', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-tag-my-posts',
				'plugin_depends' => '',
				'doc_url' => '',
			],
			'ext_tag_favorites' => [
				'name' => '',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\Favorites',
				'category' => 'POST',
				'extension_type' => 'dynamic-tag',
				'title' => __( 'Dynamic Tag - Favorites', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add support for Favorites in Dynamic Tag for text settings', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-tag-favorites',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/dynamic-tag-favorites',
			],
			'ext_tag_wishlist' => [
				'name' => '',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\WooWishlist',
				'category' => 'WOO',
				'extension_type' => 'dynamic-tag',
				'title' => __( 'Dynamic Tag - Woo Wishlist', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add support for Woo Wishlist in Dynamic Tag for text settings', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-tag-woo-wishlist',
				'plugin_depends' => [ 'woocommerce' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/dynamic-tag-woo-wishlist',
			],
			'ext_tag_acf_relationship' => [
				'name' => 'ext_tag_acf_relationship',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\AcfRelationship',
				'category' => 'ACF',
				'extension_type' => 'dynamic-tag',
				'title' => __( 'Dynamic Tag - ACF Relationship', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add support to show ACF Relationships in Dynamic Tag for text settings', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-tag-acf-relationship',
				'plugin_depends' => [ 'acf' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/dynamic-tag-acf-relationship',
			],
			'ext_tag_metabox_relationship' => [
				'name' => 'ext_tag_metabox_relationship',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\MetaboxRelationship',
				'category' => 'METABOX',
				'extension_type' => 'dynamic-tag',
				'title' => __( 'Dynamic Tag - Meta Box Relationship', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add support to show Meta Box Relationships in Dynamic Tag for text settings', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-tag-metabox-relationship',
				'plugin_depends' => [ 'meta-box' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/dynamic-tag-meta-box-relationship',
			],
			'ext_tag_terms' => [
				'name' => 'ext_tag_terms',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\Terms',
				'category' => 'TERMS',
				'extension_type' => 'dynamic-tag',
				'title' => __( 'Dynamic Tag - Terms', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add support to show terms in Dynamic Tag for text settings', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-tag-terms',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/dynamic-tag-terms',
			],
			'ext_tag_tags' => [
				'name' => 'ext_tag_tags',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\Tags',
				'category' => 'TERMS',
				'extension_type' => 'dynamic-tag',
				'title' => __( 'Dynamic Tag - Tags', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add support to show tags in Dynamic Tag for text settings', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-tag-tags',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/dynamic-tag-tags',
			],
			'ext_tag_woo_product_categories' => [
				'name' => 'ext_tag_woo_product_categories',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\WooProductCategories',
				'category' => 'WOO',
				'extension_type' => 'dynamic-tag',
				'title' => __( 'Dynamic Tag - Woo Product Categories', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add support to show WooCommerce Product Categories in Dynamic Tag for text settings', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-tag-woo-product-categories',
				'plugin_depends' => [ 'woocommerce' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/dynamic-tag-woo-product-categories',
			],
			'ext_template' => [
				'name' => 'dce_extension_template',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\Template',
				'category' => 'CONTENT',
				'extension_type' => 'dynamic-tag',
				'title' => __( 'Dynamic Tag - Template', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add support for Template in Dynamic Tag for text, HTML and textarea settings', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-template',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/dynamic-tag-template',
			],
			'ext_token' => [
				'name' => 'dce_extension_token',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\Token',
				'category' => 'CONTENT',
				'extension_type' => 'dynamic-tag',
				'title' => __( 'Dynamic Tag - Tokens', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add support for Tokens in Dynamic Tag for all settings', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-tokens',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/dynamic-tag-token/',
			],
			'dynamic_tag_php' => [
				'name' => 'dce_dynamic_tag_php',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\Php',
				'category' => 'DEV',
				'extension_type' => 'dynamic-tag',
				'title' => __( 'Dynamic Tag - PHP', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add support for PHP Code in Dynamic Tag for all settings', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-tag-php',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/dynamic-tag-php/',
			],
			'dynamic_tag_image_token' => [
				'name' => 'dce_dynamic_image_token',
				'type' => 'extension',
				'class' => 'Extensions\DynamicTags\ImageToken',
				'category' => 'CONTENT',
				'extension_type' => 'dynamic-tag',
				'title' => __( 'Dynamic Tag - Image Token', 'dynamic-content-for-elementor' ),
				'description' => __( 'A Dynamic Tag for Tokens that return Images and Media content', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-tokens',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/dynamic-tag-image-token/',
			],

			/// Widgets ///

			'wdg_acf' => [
				'class' => 'Widgets\\AcfFields',
				'type' => 'widget',
				'category' => 'ACF',
				'name' => 'dyncontel-acf',
				'title' => __( 'ACF Fields', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add a custom field created with Advanced Custom Fields', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-acffields',
				'plugin_depends' => [ 'acf' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/acf-fields/',
				'keywords' => [ 'Advanced Custom Fields' ],
			],
			'wdg_gallery' => [
				'class' => 'Widgets\\AcfGallery',
				'type' => 'widget',
				'category' => 'ACF',
				'name' => 'dyncontel-acfgallery',
				'title' => __( 'ACF Gallery', 'dynamic-content-for-elementor' ),
				'description' => __( 'Use images from an ACF Gallery field and display them in a variety of formats', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-grid',
				'plugin_depends' => [ 'advanced-custom-fields-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/acf-gallery/',
				'keywords' => [ 'Advanced Custom Fields', 'gallery', 'fields', 'images', 'image' ],
			],
			'wdg_relationship' => [
				'class' => 'Widgets\\AcfRelationshipOldVersion',
				'type' => 'widget',
				'category' => 'ACF',
				'legacy' => true,
				'replaced_by' => 'wdg_acf_relationship',
				'tag' => [ 'loop' ],
				'name' => 'dyncontel-acf-relation',
				'title' => __( 'ACF Relationship (old version)', 'dynamic-content-for-elementor' ),
				'description' => __( 'Use the ACF Relationship field to easily display related content', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-relation',
				'plugin_depends' => [ 'acf' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/acf-relationship-old-version/',
				'keywords' => [ 'Advanced Custom Fields', 'fields' ],
			],
			'wdg_acf_relationship' => [
				'class' => 'Widgets\\AcfRelationship',
				'type' => 'widget',
				'category' => 'ACF',
				'name' => 'dce-acf-relationship',
				'title' => __( 'ACF Relationship', 'dynamic-content-for-elementor' ),
				'collection' => 'dynamic-posts',
				'tag' => [ 'pagination', 'loop' ],
				'description' => __( 'Use the ACF Relationship field to easily display related content', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-relation',
				'plugin_depends' => [ 'acf' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/acf-relationship-new-version/',
				'keywords' => [ 'Advanced Custom Fields', 'fields' ],
			],
			'wdg_repeater' => [
				'class' => 'Widgets\\AcfRepeaterOldVersion',
				'type' => 'widget',
				'category' => 'ACF',
				'legacy' => true,
				'replaced_by' => 'wdg_acf_repeater_v2',
				'tag' => [ 'loop' ],
				'name' => 'dyncontel-acf-repeater',
				'title' => __( 'ACF Repeater (old version)', 'dynamic-content-for-elementor' ),
				'description' => __( 'Take advantage of the power and flexibility of the ACF Repeater field', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-repeater',
				'plugin_depends' => [ 'advanced-custom-fields-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/acf-repeater-fields/',
				'keywords' => [ 'Advanced Custom Fields' ],
			],
			'wdg_acf_repeater_v2' => [
				'class' => 'Widgets\\AcfRepeater',
				'type' => 'widget',
				'category' => 'ACF',
				'tag' => [ 'loop' ],
				'name' => 'dce-acf-repeater-v2',
				'title' => __( 'ACF Repeater', 'dynamic-content-for-elementor' ),
				'description' => __( 'Take advantage of the power and flexibility of the ACF Repeater field', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-repeater',
				'plugin_depends' => [ 'advanced-custom-fields-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/acf-repeater-fields/',
				'keywords' => [ 'Advanced Custom Fields' ],
			],
			'wdg_acf_flexible_content' => [
				'class' => 'Widgets\\AcfFlexibleContent',
				'type' => 'widget',
				'category' => 'ACF',
				'tag' => [ 'loop' ],
				'name' => 'dce-flexible-content',
				'title' => __( 'ACF Flexible Content', 'dynamic-content-for-elementor' ),
				'description' => __( 'Insert a Flexible Content field using layouts and sub fields', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-flexiblecontent',
				'plugin_depends' => [ 'advanced-custom-fields-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/acf-flexible-content/',
				'keywords' => [ 'Advanced Custom Fields' ],
			],
			'wdg_slider' => [
				'class' => 'Widgets\\AcfSlider',
				'type' => 'widget',
				'category' => 'ACF',
				'name' => 'dyncontel-acfslider',
				'title' => __( 'ACF Slider', 'dynamic-content-for-elementor' ),
				'description' => __( 'Use images from an ACF Gallery field to create an image carousel', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-slider',
				'plugin_depends' => [ 'advanced-custom-fields-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/acf-slider/',
				'keywords' => [ 'Advanced Custom Fields', 'gallery', 'images', 'carousel' ],
			],
			'wdg_barcode' => [
				'class' => 'Widgets\\QrAndBarcodes',
				'type' => 'widget',
				'category' => 'CONTENT',
				'name' => 'dce_barcode',
				'title' => __( 'QR & Barcodes', 'dynamic-content-for-elementor' ),
				'description' => __( 'Quick creation for 1D and 2D barcodes, like EAN and QR Codes', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-qrcode',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/qr-and-bars-code/',
				'keywords' => [ 'barcode', 'qr' ],
			],
			'wdg_calendar' => [
				'class' => 'Widgets\\AddToCalendar',
				'type' => 'widget',
				'category' => 'CONTENT',
				'name' => 'dce_add_to_calendar',
				'title' => __( 'Add to Calendar', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add an event to Google Calendar, iOS, Outlook, Outlook.com or Yahoo', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-buttoncalendar',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/button-calendar/',
				'keywords' => [ 'date', 'calendar', 'ics', 'reminder' ],
			],
			'wdg_clipboard' => [
				'class' => 'Widgets\\CopyToClipboard',
				'type' => 'widget',
				'category' => 'CONTENT',
				'name' => 'dce-copy-to-clipboard',
				'title' => __( 'Copy to Clipboard', 'dynamic-content-for-elementor' ),
				'description' => __( 'Copy the specified text to the User Clipboard', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-buttoncopy',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/button-copy-to-clipboard/',
			],
			'wdg_dyncookie' => [
				'class' => 'Widgets\\DynamicCookie',
				'type' => 'widget',
				'category' => 'CONTENT',
				'admin_only' => true,
				'name' => 'dce-dynamiccookie',
				'title' => __( 'Dynamic Cookie', 'dynamic-content-for-elementor' ),
				'description' => __( 'Create and manage cookies using dynamic content', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-cookie',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/dynamic-cookie',
			],
			'wdg_favorites' => [
				'class' => 'Widgets\\AddToFavorites',
				'type' => 'widget',
				'category' => 'FAVORITES',
				'name' => 'dce-add-to-favorites',
				'title' => __( 'Add to Favorites', 'dynamic-content-for-elementor' ),
				'description' => __( 'Create a favorite list for your users', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-like',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/add-to-favorites/',
				'keywords' => [ 'favourites' ],
			],
			'wdg_wishlist' => [
				'class' => 'Widgets\\AddToWooWishlist',
				'type' => 'widget',
				'category' => 'FAVORITES',
				'name' => 'dce-add-to-woo-wishlist',
				'title' => __( 'Add to Woo Wishlist', 'dynamic-content-for-elementor' ),
				'description' => __( 'Quickly and easily add WooCommerce products to the wishlist', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-woo-wishlist',
				'plugin_depends' => [ 'woocommerce' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/woocommerce-woo-add-to-wishlist/',
				'keywords' => [ 'woocommerce', 'favourites', 'favorites', 'wishlist' ],
			],
			'wdg_modal_window' => [
				'class' => 'Widgets\\FireModalWindow',
				'type' => 'widget',
				'category' => 'CONTENT',
				'name' => 'dyncontel-modalwindow',
				'title' => __( 'Fire Modal Window', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add a Modal Window button to your pages to create a popup window', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-modal',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/fire-modalwindow/',
				'keywords' => [ 'popup', 'pop-up', 'modal' ],
			],
			'wdg_pdf' => [
				'class' => 'Widgets\\PdfButton',
				'type' => 'widget',
				'category' => 'PDF',
				'admin_only' => true,
				'name' => 'dce_pdf_button',
				'title' => __( 'PDF Button', 'dynamic-content-for-elementor' ),
				'description' => __( 'Dynamically generate a PDF document using your content and styles', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-buttonpdf',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/pdf-button/',
				'keywords' => [ 'generate pdf', 'pdfs' ],
			],
			'wdg_popup' => [
				'class' => 'Widgets\\Modals',
				'type' => 'widget',
				'category' => 'CONTENT',
				'name' => 'dyncontel-popup',
				'title' => __( 'Modals', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add a modal window to your page', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-popups',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/popups/',
				'keywords' => [ 'popup', 'pop-up' ],
			],
			'wdg_charts' => [
				'class' => 'Widgets\\DynamicCharts',
				'type' => 'widget',
				'category' => 'CONTENT',
				'name' => 'dce-dynamic-charts',
				'title' => __( 'Dynamic Charts', 'dynamic-content-for-elementor' ),
				'description' => __( 'Create beautiful charts in no time at all. Plus, you can easily insert data into your charts or use a CSV file', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-charts',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/dynamic-charts/',
				'keywords' => [ 'chart', 'radar', 'doughtnut', 'pie', 'line', 'bar' ],
			],
			'wdg_template' => [
				'class' => 'Widgets\\DynamicTemplate',
				'type' => 'widget',
				'category' => 'CONTENT',
				'name' => 'dyncontel-template',
				'title' => __( 'Dynamic Template', 'dynamic-content-for-elementor' ),
				'description' => __( 'Insert an existing template and dynamically assign its content', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-layout',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/dynamic-template/',
			],
			'wdg_tokens' => [
				'class' => 'Widgets\\TextEditorWithTokens',
				'type' => 'widget',
				'category' => 'CONTENT',
				'admin_only' => true,
				'name' => 'dce-tokens',
				'title' => __( 'Text Editor with Tokens', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add Tokens to show values from posts, users, terms, custom fields, options and others', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-tokens',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/tokens/',
			],
			'wdg_discover_tokens' => [
				'class' => 'Widgets\\DiscoverTokens',
				'type' => 'widget',
				'category' => 'CONTENT',
				'admin_only' => true,
				'name' => 'dce-discover-tokens',
				'title' => __( 'Discover Tokens', 'dynamic-content-for-elementor' ),
				'description' => __( 'Discover a multitude of tokens on your site to better understand its full power', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-discover-tokens',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/discover-tokens',
				'keywords' => [ 'tokens', 'list', 'examples' ],
			],
			'wdg_text_timber' => [
				'class' => 'Widgets\\TextEditorWithTimber',
				'type' => 'widget',
				'category' => 'CONTENT',
				'name' => 'dce-text-timber',
				'admin_only' => true,
				'title' => __( 'Text Editor with Timber', 'dynamic-content-for-elementor' ),
				'description' => __( 'Write using the Twig Template Engine thanks to integration with Timber', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-text-editor-timber',
				'plugin_depends' => [ 'timber' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/text-editor-timber-twig-elementor',
				'keywords' => [ 'twig' ],
			],
			'wdg_animate_text' => [
				'class' => 'Widgets\\AnimatedText',
				'type' => 'widget',
				'category' => 'CREATIVE',
				'name' => 'dyncontel-animateText',
				'title' => __( 'Animated Text', 'dynamic-content-for-elementor' ),
				'description' => __( 'Advanced animation for your text', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-animate_text',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/animated-text/',
				'keywords' => [ 'animations' ],
			],
			'wdg_parallax' => [
				'class' => 'Widgets\\Parallax',
				'type' => 'widget',
				'category' => 'CREATIVE',
				'name' => 'dyncontel-parallax',
				'title' => __( 'Parallax', 'dynamic-content-for-elementor' ),
				'description' => __( 'Manage parallax movements with your mouse on desktop or device orientation on mobile', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-parallax',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/parallax/',
			],
			'wdg_threesixty_slider' => [
				'class' => 'Widgets\\ThreesixtySlider',
				'type' => 'widget',
				'category' => 'CREATIVE',
				'admin_only' => true,
				'name' => 'dyncontel-threesixtyslider',
				'title' => __( '360 Slider', 'dynamic-content-for-elementor' ),
				'description' => __( 'Generate a rotation through a series of images', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-360',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/360-slider/',
				'keywords' => [ 'images', 'products' ],
			],
			'wdg_tilt' => [
				'class' => 'Widgets\\Tilt',
				'type' => 'widget',
				'category' => 'CREATIVE',
				'name' => 'dyncontel-tilt',
				'title' => __( 'Tilt', 'dynamic-content-for-elementor' ),
				'description' => __( 'Parallax hover tilt effect applied to a template', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-tilt',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/tilt/',
				'keywords' => [ 'parallax' ],
			],
			'wdg_twentytwenty' => [
				'class' => 'Widgets\\BeforeAfter',
				'type' => 'widget',
				'category' => 'CREATIVE',
				'name' => 'dyncontel-twentytwenty',
				'title' => __( 'Before After', 'dynamic-content-for-elementor' ),
				'description' => __( 'Display an image with a before - after effect, ideal for comparing differences between two images', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-afterbefore',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/before-after/',
				'keywords' => [ 'images', 'image' ],
			],
			'wdg_doshortcode' => [
				'class' => 'Widgets\\ShortcodeWithTokens',
				'type' => 'widget',
				'category' => 'DEV',
				'admin_only' => true,
				'name' => 'dyncontel-doshortcode',
				'title' => __( 'Shortcode with Tokens', 'dynamic-content-for-elementor' ),
				'description' => __( 'Apply a WordPress shortcode using Tokens', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-doshortc',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/shortcode-with-tokens/',
				'keywords' => [ 'shortcode', 'tokens', 'token' ],
			],
			'wdg_includefile' => [
				'class' => 'Widgets\\FileInclude',
				'type' => 'widget',
				'category' => 'DEV',
				'admin_only' => true,
				'name' => 'dyncontel-includefile',
				'title' => __( 'File Include', 'dynamic-content-for-elementor' ),
				'description' => __( 'Directly include files from a path in root as if you were writing in a theme', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-incfile',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/file-include/',
			],
			'wdg_rawphp' => [
				'class' => 'Widgets\\PhpRaw',
				'type' => 'widget',
				'category' => 'DEV',
				'admin_only' => true,
				'name' => 'dce-rawphp',
				'title' => __( 'PHP Raw', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add PHP code directly in Elementor', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-php-raw',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/php-raw/',
				'keywords' => [ 'code' ],
			],
			'wdg_remotecontent' => [
				'class' => 'Widgets\\RemoteContent',
				'type' => 'widget',
				'category' => 'DEV',
				'admin_only' => true,
				'name' => 'dyncontel-remotecontent',
				'title' => __( 'Remote Content', 'dynamic-content-for-elementor' ),
				'description' => __( 'Dynamically read any type of content from the web, incorporate text blocks, pictures and more from external sources. Compatible with REST APIs, including the native ones from WordPress. Option to format the resulting values in JSON', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-remotecontent',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/remote-content/',
				'keywords' => [ 'iframe', 'json', 'api' ],
			],
			'wdg_iframe' => [
				'class' => 'Widgets\\Iframe',
				'type' => 'widget',
				'category' => 'CONTENT',
				'admin_only' => true,
				'name' => 'dce-iframe',
				'title' => __( 'iFrame', 'dynamic-content-for-elementor' ),
				'description' => __( 'Insert an iFrame', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-iframe',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/iframe/',
				'keywords' => [ 'iframe', 'i-frame' ],
			],
			'wdg_google_doc_preview' => [
				'class' => 'Widgets\\GoogleDocumentPreview',
				'type' => 'widget',
				'category' => 'CONTENT',
				'admin_only' => true,
				'name' => 'dce-google-document-preview',
				'title' => __( 'Document Viewer', 'dynamic-content-for-elementor' ),
				'description' => __( 'Display a preview of your document. For example: PDF, MS Office documents, Google docs, sheets, slides, forms and drawings', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-google-document-preview',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/document-viewer/',
				'keywords' => [ 'document, preview, docs' ],
			],
			'wdg_dynamicposts' => [
				'class' => 'Widgets\\DynamicPostsOldVersion',
				'type' => 'widget',
				'category' => 'LIST',
				'legacy' => true,
				'replaced_by' => 'wdg_dynamicposts_v2',
				'tag' => [ 'pagination', 'loop' ],
				'name' => 'dyncontel-acfposts',
				'title' => __( 'Dynamic Posts (old version)', 'dynamic-content-for-elementor' ),
				'description' => __( 'Create archives from lists of articles with different query source options. You can display the list with different skins and use templates to control item content', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-posts',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/dynamic-posts/',
				'keywords' => [ 'list', 'articles', 'carousel', 'slider', 'timeline', 'archive' ],
			],
			'wdg_dynamicposts_v2' => [
				
				'class' => 'Widgets\\DynamicPosts',
				'type' => 'widget',
				'category' => 'LIST',
				'collection' => 'dynamic-posts',
				'tag' => [ 'pagination', 'loop' ],
				'name' => 'dce-dynamicposts-v2',
				'title' => __( 'Dynamic Posts', 'dynamic-content-for-elementor' ),
				'description' => __( 'Create archives from lists of items with different query source options. You can display the list with various skins and you can choose which items to display', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-posts',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/dynamic-posts-v2',
				'keywords' => [ 'grid', '3d', 'skin', 'list', 'articles', 'carousel', 'slider', 'timeline', 'archive', 'custom', 'dynamic posts v2', 'custom post type', 'cpt', 'item', 'loop', 'v2', 'pods', 'acf', 'fields' ],
			],
			'wdg_dynamic_woo_products' => [
				
				'class' => 'Widgets\\DynamicWooProducts',
				'type' => 'widget',
				'category' => 'WOO',
				'collection' => 'dynamic-posts',
				'tag' => [ 'pagination', 'loop' ],
				'name' => 'dce-dynamic-woo-products',
				'title' => __( 'Dynamic Products', 'dynamic-content-for-elementor' ),
				'description' => __( 'Display your WooCommerce products with different skins and choose the items you want to display', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-woo-products',
				'plugin_depends' => [ 'woocommerce' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/dynamic-products/',
				'keywords' => [ 'woocommerce', 'woo', 'product', 'list', 'carousel', 'slider', 'timeline', 'loop' ],
			],
			'wdg_dynamic_woo_products_on_sale' => [
				
				'class' => 'Widgets\\DynamicWooProductsOnSale',
				'type' => 'widget',
				'category' => 'WOO',
				'collection' => 'dynamic-posts',
				'tag' => [ 'pagination', 'loop' ],
				'name' => 'dce-dynamic-woo-products-on-sale',
				'title' => __( 'Dynamic Products On Sale', 'dynamic-content-for-elementor' ),
				'description' => __( 'Display your WooCommerce products on sale with different skins and choose the items you want to display', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-woo-products-on-sale',
				'plugin_depends' => [ 'woocommerce' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/dynamic-products-on-sale/',
				'keywords' => [ 'woocommerce', 'woo', 'product', 'list', 'carousel', 'slider', 'timeline', 'loop', 'sale' ],
			],
			'wdg_woo_products_cart' => [
				
				'class' => 'Widgets\\WooProductsCart',
				'type' => 'widget',
				'category' => 'WOO',
				'collection' => 'dynamic-posts',
				'tag' => [ 'pagination', 'loop' ],
				'name' => 'dce-woo-products-cart',
				'title' => __( 'Products in the Cart', 'dynamic-content-for-elementor' ),
				'description' => __( 'Show your customers the WooCommerce products they have in their cart', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-woo-products-cart',
				'plugin_depends' => [ 'woocommerce' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/products-in-the-cart/',
				'keywords' => [ 'woocommerce', 'woo', 'product', 'list', 'carousel', 'slider', 'basket', 'loop' ],
			],
			'wdg_woo_product_upsells' => [
				
				'class' => 'Widgets\\WooProductUpsells',
				'type' => 'widget',
				'category' => 'WOO',
				'collection' => 'dynamic-posts',
				'tag' => [ 'pagination', 'loop' ],
				'name' => 'dce-woo-product-upsells',
				'title' => __( 'Product Upsells', 'dynamic-content-for-elementor' ),
				'description' => __( 'Show upsells products for a WooCommerce product', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-woo-product-upsells',
				'plugin_depends' => [ 'woocommerce' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/product-upsells/',
				'keywords' => [ 'woocommerce', 'woo', 'products', 'list', 'carousel', 'slider', 'loop' ],
			],
			'wdg_woo_product_crosssells' => [
				
				'class' => 'Widgets\\WooProductCrosssells',
				'type' => 'widget',
				'category' => 'WOO',
				'collection' => 'dynamic-posts',
				'tag' => [ 'pagination', 'loop' ],
				'name' => 'dce-woo-product-crosssells',
				'title' => __( 'Product Cross-Sells', 'dynamic-content-for-elementor' ),
				'description' => __( 'Show cross-sells products for a WooCommerce product', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-woo-product-crosssells',
				'plugin_depends' => [ 'woocommerce' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/product-cross-sells/',
				'keywords' => [ 'woocommerce', 'woo', 'products', 'related', 'list', 'carousel', 'slider', 'basket', 'loop' ],
			],
			'wdg_show_favorites' => [
				
				'class' => 'Widgets\\ShowFavorites',
				'type' => 'widget',
				'category' => 'FAVORITES',
				'collection' => 'dynamic-posts',
				'tag' => [ 'pagination', 'loop' ],
				'name' => 'dce-dynamic-show-favorites',
				'icon' => 'icon-dce-show-favorites',
				'title' => __( 'Favorites', 'dynamic-content-for-elementor' ),
				'description' => __( 'Show your customers the favorites they have. Works in combination with the "Add to Favorites" widget', 'dynamic-content-for-elementor' ),
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/favorites/',
				'keywords' => [ 'favourites', 'list', 'carousel', 'slider', 'loop', 'add to favorites' ],
			],
			'wdg_show_wishlist' => [
				
				'class' => 'Widgets\\WooWishlist',
				'type' => 'widget',
				'category' => 'FAVORITES',
				'collection' => 'dynamic-posts',
				'tag' => [ 'pagination', 'loop' ],
				'name' => 'dce-woo-wishlist',
				'icon' => 'icon-dce-show-wishlist',
				'title' => __( 'Woo Wishlist', 'dynamic-content-for-elementor' ),
				'description' => __( 'Show your customers their wishlist automatically, excluding bought products. Works in combination with the "Add to Woo Wishlist" widget', 'dynamic-content-for-elementor' ),
				'plugin_depends' => [ 'woocommerce' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/woocommerce-woo-wishlist/',
				'keywords' => [ 'woocommerce', 'woo', 'favourites', 'product', 'list', 'carousel', 'slider', 'basket', 'loop' ],
			],
			'wdg_stick_yposts' => [
				
				'class' => 'Widgets\\StickyPosts',
				'type' => 'widget',
				'category' => 'LIST',
				'collection' => 'dynamic-posts',
				'tag' => [ 'pagination', 'loop' ],
				'name' => 'dce-sticky-posts',
				'title' => __( 'Sticky Posts', 'dynamic-content-for-elementor' ),
				'description' => __( 'Display Sticky Posts with different skins and choose the items you want to display', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-sticky-posts',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/sticky-posts/',
				'keywords' => [ 'grid', '3d', 'skin', 'list', 'articles', 'carousel', 'slider' ],
			],
			'wdg_search_results' => [
				
				'class' => 'Widgets\\SearchResults',
				'type' => 'widget',
				'category' => 'LIST',
				'collection' => 'dynamic-posts',
				'tag' => [ 'pagination', 'loop' ],
				'name' => 'dce-search-results',
				'title' => __( 'Search Results', 'dynamic-content-for-elementor' ),
				'description' => __( 'Display Search Results with different skins and choose the items you want to display', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-search-results',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/search-results/',
				'keywords' => [ 'grid', '3d', 'skin', 'list', 'articles', 'carousel', 'slider', 'searching' ],
			],
			'wdg_my_posts' => [
				
				'class' => 'Widgets\\MyPosts',
				'type' => 'widget',
				'category' => 'LIST',
				'collection' => 'dynamic-posts',
				'tag' => [ 'pagination', 'loop' ],
				'name' => 'dce-my-posts',
				'title' => __( 'Posts by the Current User', 'dynamic-content-for-elementor' ),
				'description' => __( 'Show your logged-in customers posts written by them', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-my-posts',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/my-posts/',
				'keywords' => [ 'grid', '3d', 'skin', 'list', 'articles', 'carousel', 'slider', 'searching', 'current author', 'author' ],
			],
			'wdg_metabox_relationship' => [
				
				'class' => 'Widgets\\MetaboxRelationship',
				'type' => 'widget',
				'category' => 'METABOX',
				'collection' => 'dynamic-posts',
				'tag' => [ 'pagination', 'loop' ],
				'name' => 'dce-metabox-relationship',
				'title' => __( 'Meta Box Relationship', 'dynamic-content-for-elementor' ),
				'description' => __( 'Use Meta Box Relationships to easily display related content', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-metabox-relationship',
				'plugin_depends' => [ 'meta-box' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/meta-box-relationship-elementor/',
				'keywords' => [ 'metabox' ],
			],
			'wdg_google_maps' => [
				'class' => 'Widgets\\DynamicGoogleMaps',
				'type' => 'widget',
				'category' => 'MAPS',
				'admin_only' => true,
				'name' => 'dyncontel-acf-google-maps',
				'title' => __( 'Dynamic Google Maps', 'dynamic-content-for-elementor' ),
				'description' => __( 'Build a map using data from ACF Google Map fields, Meta Box Google Map fields, addresses or latitude and longitude', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-map',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/dynamic-google-maps/',
				'keywords' => [ 'Advanced Custom Fields', 'fields' ],
			],
			'wdg_google_maps_directions' => [
				'class' => 'Widgets\\DynamicGoogleMapsDirections',
				'type' => 'widget',
				'category' => 'MAPS',
				'name' => 'dce-dynamic-google-maps-directions',
				'title' => __( 'Dynamic Google Maps Directions', 'dynamic-content-for-elementor' ),
				'description' => __( 'Show the map and calculate the direction between two locations. Can be used in conjunction with its Dynamic Tags to show distance info and directions', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-google-maps-directions',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/dynamic-google-maps-directions',
				'keywords' => [ 'distance' ],
			],
			'wdg_metabox_google_maps' => [
				'class' => 'Widgets\\MetaBoxGoogleMaps',
				'type' => 'widget',
				'category' => 'METABOX',
				'name' => 'dce-metabox-google-maps',
				'title' => __( 'Meta Box Google Maps', 'dynamic-content-for-elementor' ),
				'description' => __( 'Build a map using data from Meta Box Google Map fields', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-metabox-google-maps',
				'plugin_depends' => [ 'meta-box' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/meta-box-google-maps-elementor/',
				'keywords' => [ 'metabox', 'fields' ],
			],
			'wdg_dynamic_users' => [
				'class' => 'Widgets\\DynamicUsers',
				'type' => 'widget',
				'category' => 'LIST',
				'admin_only' => true,
				'tag' => [ 'pagination', 'loop' ],
				'name' => 'dyncontel-dynamicusers',
				'title' => __( 'Dynamic Users', 'dynamic-content-for-elementor' ),
				'description' => __( 'Create an archive based on User profiles', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-users',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/dynamic-users/',
				'keywords' => [ 'list', 'users', 'archive' ],
			],
			'wdg_animated_offcanvas_menu' => [
				'class' => 'Widgets\\AnimatedOffCanvasMenu',
				'type' => 'widget',
				'category' => 'CREATIVE',
				'name' => 'dce-animatedoffcanvasmenu',
				'title' => __( 'Animated Off-Canvas Menu', 'dynamic-content-for-elementor' ),
				'description' => __( 'An off-canvas menu is positioned outside of the viewport and slides in when activated', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-animatedoffcanvasmenu',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/animated-offcanvas-menu/',
				'keywords' => [ 'menu', 'animations' ],
			],
			'wdg_cursor_tracker' => [
				'class' => 'Widgets\\CursorTracker',
				'type' => 'widget',
				'category' => 'CREATIVE',
				'name' => 'dyncontel-cursorTracker',
				'title' => __( 'Cursor Tracker', 'dynamic-content-for-elementor' ),
				'description' => __( 'An element that follows the cursor and indicates the level of scrolling', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-cursor-tracker',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/cursor-tracker/',
			],
			'wdg_file_browser' => [
				'class' => 'Widgets\\FileBrowser',
				'type' => 'widget',
				'category' => 'LIST',
				'admin_only' => true,
				'name' => 'dce-filebrowser',
				'title' => __( 'FileBrowser', 'dynamic-content-for-elementor' ),
				'description' => __( 'Display a list of files you uploaded. This is particularly useful when you need to make pictures or documents available in a simple and intuitive way', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-filebrowser',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/file-browser/',
				'keywords' => [ 'list', 'file', 'download', 'archive' ],
			],
			'wdg_parent_child_menu' => [
				'class' => 'Widgets\\ParentChildMenu',
				'type' => 'widget',
				'category' => 'LIST',
				'name' => 'parent-child-menu',
				'title' => __( 'Parent Child Menu', 'dynamic-content-for-elementor' ),
				'description' => __( 'Build a list of entries in horizontal or vertical mode where a parent element can be considered as a menu', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-parentchild',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/widget-parent-child-menu/',
			],
			'wdg_search_filter' => [
				'class' => 'Widgets\\AdvancedFilteringSearchAndFilterPro',
				'type' => 'widget',
				'category' => 'LIST',
				'name' => 'dce-searchfilter',
				'title' => __( 'Advanced Filtering from Search & Filter Pro', 'dynamic-content-for-elementor' ),
				'description' => __( 'The Ultimate WordPress filter plugin that uses Ajax with Elementor', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-search-filter',
				'plugin_depends' => [ 'search-filter-pro' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/search-filter-pro-elementor',
				'keywords' => [ 'filters', 'Search and Filter Pro' ],
			],
			'wdg_single_posts_menu' => [
				'class' => 'Widgets\\SinglePostsList',
				'type' => 'widget',
				'category' => 'LIST',
				'name' => 'single-posts-menu',
				'title' => __( 'Single Posts List', 'dynamic-content-for-elementor' ),
				'description' => __( 'Create a custom menu from single pages', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-listsingle',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/single-posts-list/',
			],
			'wdg_taxonomy_terms_menu' => [
				'class' => 'Widgets\\TaxonomyTermsList',
				'type' => 'widget',
				'category' => 'LIST',
				'name' => 'taxonomy-terms-menu',
				'title' => __( 'Taxonomy Terms List', 'dynamic-content-for-elementor' ),
				'description' => __( 'Create an archive type navigation menu based on a taxonomy. It generates a list of the terms that belong to a specific taxonomy so that each item can be clicked to easily access to the term archive', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-parenttax',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/taxonomy-terms-list/',
				'keywords' => [ 'list', 'menu' ],
			],
			'wdg_views' => [
				'class' => 'Widgets\\Views',
				'type' => 'widget',
				'category' => 'LIST',
				'admin_only' => true,
				'name' => 'dce-views',
				'tag' => [ 'loop' ],
				'title' => __( 'Views', 'dynamic-content-for-elementor' ),
				'description' => __( 'Create a custom list from query results', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-views',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/views/',
				'keywords' => [ 'list', 'articles', 'search', 'archive', 'cpt' ],
			],
			'wdg_pods' => [
				'class' => 'Widgets\\PodsFields',
				'type' => 'widget',
				'category' => 'PODS',
				'name' => 'dyncontel-pods',
				'title' => __( 'Pods Fields', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add a custom field created with Pods', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-acffields',
				'plugin_depends' => [ 'pods' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/pods-fields/',
			],
			'wdg_pods_gallery' => [
				'class' => 'Widgets\\PodsGallery',
				'type' => 'widget',
				'category' => 'PODS',
				'name' => 'dyncontel-podsgallery',
				'title' => __( 'Pods Gallery', 'dynamic-content-for-elementor' ),
				'description' => __( 'Use a list of images from a Pods Image field with various display options', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-grid',
				'plugin_depends' => [ 'pods' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/pods-gallery/',
				'keywords' => [ 'images', 'fields' ],
			],
			'wdg_pods_relationship' => [
				'class' => 'Widgets\\PodsRelationship',
				'type' => 'widget',
				'category' => 'PODS',
				'admin_only' => true,
				'tag' => [ 'loop' ],
				'name' => 'dyncontel-pods-relation',
				'title' => __( 'Pods Relationship', 'dynamic-content-for-elementor' ),
				'description' => __( 'Display related posts linked using the Pods Relationship field', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-relation',
				'plugin_depends' => [ 'pods' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/pods-relationship/',
			],
			'wdg_breadcrumbs' => [
				'class' => 'Widgets\\Breadcrumbs',
				'type' => 'widget',
				'category' => 'POST',
				'name' => 'dce-breadcrumbs',
				'title' => __( 'Breadcrumbs', 'dynamic-content-for-elementor' ),
				'description' => __( 'Insert breadcrumbs and generate paths inside your page automatically', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-breadcrumbs',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/breadcrumbs/',
			],
			'wdg_content' => [
				'class' => 'Widgets\\Content',
				'type' => 'widget',
				'category' => 'POST',
				'name' => 'dyncontel-content',
				'title' => __( 'Content', 'dynamic-content-for-elementor' ),
				'description' => __( 'Retrieve the post content', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-content',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/content/',
			],
			'wdg_date' => [
				'class' => 'Widgets\\Date',
				'type' => 'widget',
				'category' => 'POST',
				'name' => 'dyncontel-date',
				'title' => __( 'Date', 'dynamic-content-for-elementor' ),
				'description' => __( 'Display the published or modified date', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-date',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/date/',
			],
			'wdg_excerpt' => [
				'class' => 'Widgets\\Excerpt',
				'type' => 'widget',
				'category' => 'POST',
				'name' => 'dyncontel-excerpt',
				'title' => __( 'Excerpt', 'dynamic-content-for-elementor' ),
				'description' => __( 'Display a short section from the beginning of the content', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-excerpt',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/excerpt/',
				'keywords' => [ 'content' ],
			],
			'wdg_featured_image' => [
				'class' => 'Widgets\\FeaturedImage',
				'type' => 'widget',
				'category' => 'POST',
				'name' => 'dyncontel-featured-image',
				'title' => __( 'Featured Image', 'dynamic-content-for-elementor' ),
				'description' => __( 'Retrieve the featured image', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-featured-image',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/featured-image/',
			],
			'wdg_icon_format' => [
				'class' => 'Widgets\\IconFormat',
				'type' => 'widget',
				'category' => 'POST',
				'name' => 'dyncontel-iconformat',
				'title' => __( 'Icon Format', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add an icon for your post format and identify its type', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-formats',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/icon-format/',
			],
			'wdg_meta' => [
				'class' => 'Widgets\\PostMeta',
				'type' => 'widget',
				'category' => 'POST',
				'admin_only' => true,
				'name' => 'dce-meta',
				'title' => __( 'Post Meta', 'dynamic-content-for-elementor' ),
				'description' => __( 'Display any post meta saved in the database', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-customfields',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/post-meta/',
				'legacy' => true,
			],
			'wdg_next_prev' => [
				'class' => 'Widgets\\PrevNext',
				'type' => 'widget',
				'category' => 'POST',
				'name' => 'dyncontel-post-nextprev',
				'title' => __( 'Prev Next', 'dynamic-content-for-elementor' ),
				'description' => __( 'Display navigation links to adjacent posts based on a category/taxonomy or tag', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-prevnext',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/prevnext/',
				'keywords' => [ 'previous', 'list' ],
			],
			'wdg_read_more' => [
				'class' => 'Widgets\\ReadMore',
				'type' => 'widget',
				'category' => 'POST',
				'name' => 'dyncontel-readmore',
				'title' => __( 'Read More Button', 'dynamic-content-for-elementor' ),
				'description' => __( 'Add a "Read More" button below your post or on the block in the archive, create a call-to-action.', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-readmore',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/read-more-button/',
			],
			'wdg_terms' => [
				'class' => 'Widgets\\TermsAndTaxonomy',
				'type' => 'widget',
				'category' => 'POST',
				'name' => 'dyncontel-terms',
				'title' => __( 'Terms & Taxonomy', 'dynamic-content-for-elementor' ),
				'description' => __( 'Insert your post taxonomies', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-terms',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/terms-and-taxonomy/',
				'keywords' => [ 'taxonomies' ],
			],
			'wdg_title' => [
				'class' => 'Widgets\\DynamicTitle',
				'type' => 'widget',
				'category' => 'POST',
				'name' => 'dyncontel-title',
				'title' => __( 'Dynamic Title', 'dynamic-content-for-elementor' ),
				'description' => __( 'It automatically recognises the object among Posts, Archives and Users and gives you the title', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-title',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/title/',
			],
			'wdg_title_type' => [
				'class' => 'Widgets\\PostType',
				'type' => 'widget',
				'category' => 'POST',
				'name' => 'dyncontel-titleType',
				'title' => __( 'Post Type', 'dynamic-content-for-elementor' ),
				'description' => __( 'Retrieve the post type', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-title-type',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/title-type/',
				'keywords' => [ 'cpt', 'Custom Post Type' ],
			],
			'wdg_user' => [
				'class' => 'Widgets\\UserFields',
				'type' => 'widget',
				'category' => 'POST',
				'admin_only' => true,
				'name' => 'dce-user-fields',
				'title' => __( 'User Fields', 'dynamic-content-for-elementor' ),
				'description' => __( 'Display any user field', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-userfields',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/user-fields/',
			],
			'wdg_title_taxonomy' => [
				'class' => 'Widgets\\TaxonomyTitle',
				'type' => 'widget',
				'category' => 'POST',
				'name' => 'dyncontel-titleTaxonomy',
				'title' => __( 'Taxonomy Title', 'dynamic-content-for-elementor' ),
				'description' => __( 'Use the current Taxonomy term as the Archive title', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-title-taxonomy',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/title-taxonomy/',
			],
			'wdg_svg_blob' => [
				'class' => 'Widgets\\SvgBlob',
				'type' => 'widget',
				'category' => 'SVG',
				'admin_only' => true,
				'name' => 'dyncontel-svgblob',
				'title' => __( 'SVG Blob', 'dynamic-content-for-elementor' ),
				'description' => __( 'Create a shape using an SVG path and make it move!', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-svgblob',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/svg-blob/',
				'keywords' => [ 'animations' ],
			],
			'wdg_svg_distortion' => [
				'class' => 'Widgets\\SvgDistortion',
				'type' => 'widget',
				'category' => 'SVG',
				'admin_only' => true,
				'name' => 'dyncontel-svgdistortion',
				'title' => __( 'SVG Distortion', 'dynamic-content-for-elementor' ),
				'description' => __( 'The SVG Distortion widget operates by calculating the displacement map based on a source image', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-distortion',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/svg-distortion/',
				'keywords' => [ 'animations' ],
			],
			'wdg_svg_filter_effects' => [
				'class' => 'Widgets\\SvgFilterEffects',
				'type' => 'widget',
				'category' => 'SVG',
				'admin_only' => true,
				'name' => 'dyncontel-filtereffects',
				'title' => __( 'SVG Filter Effects', 'dynamic-content-for-elementor' ),
				'description' => __( 'The widget operates using primitive SVG filters', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-svgfe',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/svg-fe-filtereffects/',
			],
			'wdg_svg_imagemask' => [
				'class' => 'Widgets\\SvgImageMask',
				'type' => 'widget',
				'category' => 'SVG',
				'admin_only' => true,
				'name' => 'dyncontel-svgimagemask',
				'title' => __( 'SVG Image Mask', 'dynamic-content-for-elementor' ),
				'description' => __( 'The SVG Image Mask widget operates through the mask attribute of the SVG', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-svgmask',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/svg-mask/',
				'keywords' => [ 'masking', 'images' ],
			],
			'wdg_svg_morphing' => [
				'class' => 'Widgets\\SvgMorphing',
				'type' => 'widget',
				'category' => 'SVG',
				'admin_only' => true,
				'name' => 'dyncontel-svgmorphing',
				'title' => __( 'SVG Morphing', 'dynamic-content-for-elementor' ),
				'description' => __( 'You can transpose SVG paths to morph from one shape to another ', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-svgmorph',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/svg-morphing/',
				'keywords' => [ 'animations' ],
			],
			'wdg_svg_path_text' => [
				'class' => 'Widgets\\SvgTextPath',
				'type' => 'widget',
				'category' => 'SVG',
				'admin_only' => true,
				'name' => 'dyncontel-svgpathText',
				'title' => __( 'SVG Text Path', 'dynamic-content-for-elementor' ),
				'description' => __( 'Write out text along a source based on an SVG path line', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-svgtextpath',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/svg-text-path/',
			],
			'wdg_toolset' => [
				'class' => 'Widgets\\ToolsetFields',
				'type' => 'widget',
				'category' => 'TOOLSET',
				'name' => 'dyncontel-toolset',
				'title' => __( 'Toolset Fields', 'dynamic-content-for-elementor' ),
				'description' => __( 'Display a Toolset custom field', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-acffields',
				'plugin_depends' => [ 'types' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/toolset-fields/',
			],
			'wdg_toolset_relationship' => [
				'class' => 'Widgets\\ToolsetRelationship',
				'type' => 'widget',
				'category' => 'TOOLSET',
				'admin_only' => true,
				'tag' => [ 'loop' ],
				'name' => 'dyncontel-toolset-relation',
				'title' => __( 'Toolset Relationship', 'dynamic-content-for-elementor' ),
				'description' => __( 'Display related posts defined by Toolset Relationships', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-relation',
				'plugin_depends' => [ 'types' ],
				'doc_url' => 'https://www.dynamic.ooo/widget/toolset-relationship/',
				'keywords' => [ 'fields' ],
			],
			'wdg_bg_canvas' => [
				'class' => 'Widgets\\BgCanvas',
				'type' => 'widget',
				'category' => 'WEBGL',
				'name' => 'dyncontel-bgcanvas',
				'title' => __( 'Background Canvas', 'dynamic-content-for-elementor' ),
				'description' => __( 'Easily integrate WebGL with Canvas for Shader effects', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-canvas',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/webgl-threejs-background-canvas/',
				'keywords' => [ 'bg', 'images' ],
			],
			'wdg_distortion_image' => [
				'class' => 'Widgets\\ImagesDistortionHover',
				'type' => 'widget',
				'category' => 'WEBGL',
				'name' => 'dyncontel-imagesDistortion',
				'title' => __( 'Images Distortion Hover', 'dynamic-content-for-elementor' ),
				'description' => __( 'Special rollover effects based on WebGL, Three.js, and transformation shaders', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-distortion',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/webgl-threejs-images-distortions-hover/',
				'keywords' => [ 'animations' ],
			],
			'wdg_panorama' => [
				'class' => 'Widgets\\Panorama',
				'type' => 'widget',
				'category' => 'WEBGL',
				'name' => 'dyncontel-panorama',
				'title' => __( 'Panorama', 'dynamic-content-for-elementor' ),
				'description' => __( 'Display a 360 degree spherical image through VR mode', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-panorama',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/panorama/',
				'keywords' => [ 'images' ],
			],
			'wdg_osm_map' => [
				'class' => 'Widgets\\OsmMap',
				'type' => 'widget',
				'category' => 'MAPS',
				'name' => 'dce-osm-map',
				'title' => __( 'OpenStreetMap', 'dynamic-content-for-elementor' ),
				'description' => __( 'A map widget that does not need a key thanks to OpenStreetMap!', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-osm',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/openstreetmap-elementor/',
				'keywords' => [ 'map', 'open street map', 'google maps', 'osm', 'address' ],
			],
			'wdg_dynamic_osm_map' => [
				'class' => 'Widgets\\DynamicOsmMap',
				'type' => 'widget',
				'category' => 'MAPS',
				'name' => 'dce-dynamic-osm-map',
				'title' => __( 'Dynamic OpenStreetMap', 'dynamic-content-for-elementor' ),
				'description' => __( 'Display and customize multiple points of interest on OpenStreetMap', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-dynamic-openstreetmap',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/dynamic-openstreetmap/',
				'keywords' => [ 'map', 'open street map', 'google maps', 'osm', 'address' ],
			],
			'wdg_cryptocoin_badge' => [
				'class' => 'Widgets\\CryptocoinBadge',
				'type' => 'widget',
				'category' => 'CONTENT',
				'name' => 'dce-cryptocoin-badge',
				'title' => __( 'Cryptocoin Badge', 'dynamic-content-for-elementor' ),
				'description' => __( 'Perfect for showing your customers the realtime value of your favorite cryptocoin', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-crypto-badge',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/cryptocoin-badge',
				'keywords' => [ 'realtime', 'bitcoin', 'crypto' ],
			],
			'wdg_pdf_viewer' => [
				'class' => 'Widgets\\PdfViewer',
				'type' => 'widget',
				'category' => 'PDF',
				'name' => 'dce-pdf-viewer',
				'title' => __( 'PDF Viewer', 'dynamic-content-for-elementor' ),
				'description' => __( 'Show a PDF file', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dce-pdf-viewer',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/pdf-viewer/',
				'keywords' => [ 'pdf', 'viewer', 'preview' ],
			],

			// Page Settings

			'pst_scrolling' => [
				'class' => 'PageSettings\\PageScroll',
				'type' => 'page-setting',
				'name' => 'dce_document_scrolling',
				'title' => __( 'Page Scroll', 'dynamic-content-for-elementor' ),
				'description' => __( 'Turn sections into scrolling steps and minimize pagination using either scroll, snap, or inertia effects', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-page-scroll',
				'plugin_depends' => '',
				'doc_url' => 'https://www.dynamic.ooo/widget/page-scroll-document/',
			],

			// Global Settings

			'gst_smooth_transition' => [
				'class' => 'Includes\\Settings\\SmoothTransition',
				'type' => 'global-setting',
				'name' => 'dce_settings_smoothtransition',
				'title' => __( 'Smooth Transition', 'dynamic-content-for-elementor' ),
				'description' => __( 'Provides a smooth animated transition between pages', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-smooth-transition',
				'plugin_depends' => [],
				'doc_url' => 'https://www.dynamic.ooo/widget/global-settings-smooth-transition/',
				'legacy' => true,
			],
			'gst_tracker_header' => [
				'class' => 'Includes\\Settings\\TrackerHeader',
				'type' => 'global-setting',
				'name' => 'dce_settings_trackerheader',
				'title' => __( 'Tracker Header', 'dynamic-content-for-elementor' ),
				'description' => __( 'Tracker header will fix the header of your site to the top of the page. The header will disappear while scrolling but will reappear with a small swipe upward. This will help to optimize available spaces, especially on mobile devices', 'dynamic-content-for-elementor' ),
				'icon' => 'icon-dyn-tracker-header',
				'plugin_depends' => [],
				'doc_url' => 'https://www.dynamic.ooo/widget/global-settings-tracker-header/',
			],
		];
	}
}
